// Products Management with MongoDB API
const API_BASE_URL = window.location.origin;
const ADMIN_MODE_KEY = 'admin_mode';

// API Functions
async function getProducts(filter = 'all') {
    try {
        const url = filter === 'all' 
            ? `${API_BASE_URL}/api/products`
            : `${API_BASE_URL}/api/products?category=${filter}`;
        
        console.log('Fetching products from:', url);
        
        const response = await fetch(url);
        
        if (!response.ok) {
            console.error('API Error:', response.status, response.statusText);
            throw new Error(`Failed to fetch products: ${response.status}`);
        }
        
        const data = await response.json();
        console.log('Products loaded:', data.length);
        return data;
    } catch (error) {
        console.error('Error fetching products:', error);
        console.error('API Base URL:', API_BASE_URL);
        return [];
    }
}

// Check if admin mode is enabled
function isAdminMode() {
    return localStorage.getItem(ADMIN_MODE_KEY) === 'true';
}

// Display products
async function displayProducts(filter = 'all') {
    const productsGrid = document.getElementById('productsGrid');
    const emptyState = document.getElementById('emptyState');
    
    if (!productsGrid) {
        console.error('Products grid not found!');
        return;
    }

    // Show loading state
    productsGrid.innerHTML = '<div style="grid-column: 1/-1; text-align: center; padding: 40px;"><i class="fas fa-spinner fa-spin" style="font-size: 40px; color: var(--primary-teal);"></i><p style="margin-top: 20px;">جاري تحميل المنتجات...</p></div>';

    try {
        // Fetch products from API
        const products = await getProducts(filter);

        // Clear grid
        productsGrid.innerHTML = '';

        if (products.length === 0) {
            productsGrid.style.display = 'none';
            if (emptyState) {
                emptyState.style.display = 'block';
                emptyState.innerHTML = `
                    <i class="fas fa-box-open"></i>
                    <p>لا توجد منتجات متاحة حالياً</p>
                    <p style="font-size: 14px; color: #999; margin-top: 10px;">
                        قم بإضافة منتجات من <a href="/admin" style="color: var(--primary-teal); text-decoration: underline;">لوحة التحكم</a>
                    </p>
                `;
            }
            console.log('No products found in database');
            return;
        }

        productsGrid.style.display = 'grid';
        if (emptyState) emptyState.style.display = 'none';

        // Display products
        products.forEach((product, index) => {
            const productCard = createProductCard(product, index);
            productsGrid.appendChild(productCard);
        });
    } catch (error) {
        console.error('Error displaying products:', error);
        productsGrid.innerHTML = `
            <div style="grid-column: 1/-1; text-align: center; padding: 40px; color: #dc3545;">
                <i class="fas fa-exclamation-triangle" style="font-size: 40px; margin-bottom: 20px;"></i>
                <p>حدث خطأ في تحميل المنتجات</p>
                <p style="font-size: 14px; margin-top: 10px;">تأكد من تشغيل السيرفر</p>
                <button onclick="location.reload()" style="margin-top: 20px; padding: 10px 20px; background: var(--primary-teal); color: white; border: none; border-radius: 5px; cursor: pointer;">
                    إعادة المحاولة
                </button>
            </div>
        `;
    }
}

// Create product card element
function createProductCard(product, index) {
    const card = document.createElement('div');
    card.className = 'product-card';
    card.dataset.index = index;

    const categoryNames = {
        'volcanic': 'البلك البركاني'
    };

    const specsHTML = product.specs && product.specs.length > 0
        ? `<ul class="product-specs">
            ${product.specs.map(spec => `<li><i class="fas fa-check"></i> ${spec}</li>`).join('')}
           </ul>`
        : '';

    card.innerHTML = `
        <div class="product-image" onclick="viewProductDetails('${product._id}')">
            ${product.image 
                ? `<img src="${product.image}" alt="${product.name}" onerror="this.parentElement.innerHTML='<div class=\\'placeholder\\'><i class=\\'fas fa-box\\'></i></div>'">`
                : `<div class="placeholder"><i class="fas fa-box"></i></div>`
            }
        </div>
        <div class="product-info">
            <span class="product-category">${categoryNames[product.category] || product.category}</span>
            <h3 class="product-name" onclick="viewProductDetails('${product._id}')">${product.name}</h3>
            ${product.description ? `<p class="product-description">${product.description}</p>` : ''}
            ${specsHTML}
            <div class="product-price">${product.price} ريال</div>
            <div class="product-actions">
                <button class="btn-contact" onclick="event.stopPropagation(); viewProductDetails('${product._id}')">
                    <i class="fas fa-phone"></i> تواصل معنا
                </button>
            </div>
        </div>
    `;
    
    // Make entire card clickable
    card.addEventListener('click', (e) => {
        if (!e.target.closest('.btn-contact')) {
            viewProductDetails(product._id);
        }
    });

    return card;
}

// Contact about product
function contactAboutProduct(productName) {
    const message = `أريد الاستفسار عن: ${productName}`;
    const phone = '+966561033147';
    const whatsappLink = `https://wa.me/${phone.replace(/\D/g, '')}?text=${encodeURIComponent(message)}`;
    window.open(whatsappLink, '_blank');
}

// Navigate to product details
function viewProductDetails(productId) {
    window.location.href = `product-details.html?id=${productId}`;
}

// Initialize page
document.addEventListener('DOMContentLoaded', () => {
    console.log('Products page loaded');
    console.log('API Base URL:', API_BASE_URL);

    // Display products (only volcanic)
    displayProducts('volcanic');
});

// Export for use in other files
window.getProducts = getProducts;

