const express = require('express');
const mongoose = require('mongoose');
const cors = require('cors');
const session = require('express-session');
const bcrypt = require('bcrypt');
const path = require('path');
require('dotenv').config();

const app = express();
const PORT = process.env.PORT || 3000;

// Live Reload for development
if (process.env.NODE_ENV !== 'production') {
    try {
        const livereload = require('livereload');
        const connectLivereload = require('connect-livereload');
        
        // Function to find available port
        function findAvailablePort(startPort, callback) {
            const net = require('net');
            const server = net.createServer();
            
            server.listen(startPort, () => {
                const port = server.address().port;
                server.close(() => {
                    callback(port);
                });
            });
            
            server.on('error', () => {
                findAvailablePort(startPort + 1, callback);
            });
        }
        
        findAvailablePort(35729, (livereloadPort) => {
            const liveReloadServer = livereload.createServer({
                extraExts: ['html', 'css', 'js'],
                port: livereloadPort,
                delay: 1000
            });
            
            liveReloadServer.watch([
                path.join(__dirname, 'public'),
                path.join(__dirname, 'public/css'),
                path.join(__dirname, 'public/js'),
                path.join(__dirname, 'public/img')
            ]);
            
            liveReloadServer.server.once('connection', () => {
                setTimeout(() => {
                    liveReloadServer.refresh('/');
                }, 100);
            });
            
            // Handle Live Reload server errors
            liveReloadServer.server.on('error', (err) => {
                if (err.code === 'EADDRINUSE') {
                    console.log('⚠️  Live Reload port conflict. Trying different port...');
                } else {
                    console.error('Live Reload error:', err.message);
                }
            });
            
            app.use(connectLivereload({
                port: livereloadPort
            }));
            
            console.log(`✅ Live Reload enabled on port ${livereloadPort}`);
        });
    } catch (error) {
        console.log('⚠️  Live Reload not available:', error.message);
    }
}

// Session configuration
app.use(session({
    secret: process.env.SESSION_SECRET || 'your-secret-key-change-this-in-production',
    resave: false,
    saveUninitialized: false,
    cookie: {
        secure: process.env.NODE_ENV === 'production', // HTTPS in production
        httpOnly: true,
        maxAge: 24 * 60 * 60 * 1000 // 24 hours
    }
}));

// Middleware
app.use(cors({
    origin: true,
    credentials: true
}));
app.use(express.json());
app.use(express.urlencoded({ extended: true }));
// Disable caching completely (for development and production)
app.use((req, res, next) => {
    // Disable all caching
    res.set({
        'Cache-Control': 'no-store, no-cache, must-revalidate, proxy-revalidate, max-age=0',
        'Pragma': 'no-cache',
        'Expires': '0',
        'Surrogate-Control': 'no-store',
        'X-Content-Type-Options': 'nosniff'
    });
    
    // Remove ETag
    res.removeHeader('ETag');
    
    next();
});

app.use(express.static('public', {
    etag: false,
    lastModified: false,
    setHeaders: (res, path) => {
        // Force no cache for all static files
        res.set({
            'Cache-Control': 'no-store, no-cache, must-revalidate, proxy-revalidate, max-age=0',
            'Pragma': 'no-cache',
            'Expires': '0'
        });
    }
})); // Serve static files

// MongoDB Connection
const MONGODB_URI = process.env.MONGODB_URI || 'mongodb://localhost:27017/jumhour';

mongoose.connect(MONGODB_URI, {
    useNewUrlParser: true,
    useUnifiedTopology: true,
})
.then(() => {
    console.log('✅ Connected to MongoDB');
})
.catch((error) => {
    console.error('❌ MongoDB connection error:', error);
});

// Product Schema
const productSchema = new mongoose.Schema({
    name: {
        type: String,
        required: true,
        trim: true
    },
    category: {
        type: String,
        required: true,
        enum: ['volcanic'],
        default: 'volcanic'
    },
    description: {
        type: String,
        default: ''
    },
    price: {
        type: Number,
        required: true,
        min: 0
    },
    image: {
        type: String,
        default: ''
    },
    specs: {
        type: [String],
        default: []
    },
    createdAt: {
        type: Date,
        default: Date.now
    },
    updatedAt: {
        type: Date,
        default: Date.now
    }
});

const Product = mongoose.model('Product', productSchema);

// Admin Schema
const adminSchema = new mongoose.Schema({
    username: {
        type: String,
        required: true,
        unique: true,
        trim: true
    },
    password: {
        type: String,
        required: true
    },
    createdAt: {
        type: Date,
        default: Date.now
    }
});

const Admin = mongoose.model('Admin', adminSchema);

// Initialize default admin (run once)
async function initializeAdmin() {
    try {
        const adminCount = await Admin.countDocuments();
        if (adminCount === 0) {
            const defaultUsername = process.env.ADMIN_USERNAME || 'admin';
            const defaultPassword = process.env.ADMIN_PASSWORD || 'admin123';
            const hashedPassword = await bcrypt.hash(defaultPassword, 10);
            
            await Admin.create({
                username: defaultUsername,
                password: hashedPassword
            });
            console.log('✅ Default admin created');
            console.log(`   Username: ${defaultUsername}`);
            console.log(`   Password: ${defaultPassword}`);
            console.log('⚠️  Please change the default password!');
        }
    } catch (error) {
        console.error('Error initializing admin:', error);
    }
}

// Authentication Middleware
function requireAuth(req, res, next) {
    if (req.session && req.session.isAuthenticated) {
        return next();
    }
    res.status(401).json({ error: 'Unauthorized' });
}

// Login route
app.post('/api/auth/login', async (req, res) => {
    try {
        const { username, password } = req.body;
        
        if (!username || !password) {
            return res.status(400).json({ error: 'Username and password are required' });
        }
        
        const admin = await Admin.findOne({ username });
        if (!admin) {
            return res.status(401).json({ error: 'Invalid credentials' });
        }
        
        const isValidPassword = await bcrypt.compare(password, admin.password);
        if (!isValidPassword) {
            return res.status(401).json({ error: 'Invalid credentials' });
        }
        
        req.session.isAuthenticated = true;
        req.session.username = admin.username;
        
        res.json({ 
            message: 'Login successful',
            username: admin.username
        });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

// Logout route
app.post('/api/auth/logout', (req, res) => {
    req.session.destroy((err) => {
        if (err) {
            return res.status(500).json({ error: 'Error logging out' });
        }
        res.json({ message: 'Logout successful' });
    });
});

// Check auth status
app.get('/api/auth/status', (req, res) => {
    res.json({ 
        isAuthenticated: req.session && req.session.isAuthenticated || false,
        username: req.session && req.session.username || null
    });
});

// Change password route
app.post('/api/auth/change-password', requireAuth, async (req, res) => {
    try {
        const { currentPassword, newPassword } = req.body;
        
        if (!currentPassword || !newPassword) {
            return res.status(400).json({ error: 'Current and new passwords are required' });
        }
        
        const admin = await Admin.findOne({ username: req.session.username });
        if (!admin) {
            return res.status(404).json({ error: 'Admin not found' });
        }
        
        const isValidPassword = await bcrypt.compare(currentPassword, admin.password);
        if (!isValidPassword) {
            return res.status(401).json({ error: 'Current password is incorrect' });
        }
        
        const hashedPassword = await bcrypt.hash(newPassword, 10);
        admin.password = hashedPassword;
        await admin.save();
        
        res.json({ message: 'Password changed successfully' });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

// API Routes

// Get all products
app.get('/api/products', async (req, res) => {
    try {
        const { category } = req.query;
        const filter = category && category !== 'all' ? { category } : {};
        const products = await Product.find(filter).sort({ createdAt: -1 });
        res.json(products);
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

// Get single product
app.get('/api/products/:id', async (req, res) => {
    try {
        const product = await Product.findById(req.params.id);
        if (!product) {
            return res.status(404).json({ error: 'Product not found' });
        }
        res.json(product);
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

// Create product (protected)
app.post('/api/products', requireAuth, async (req, res) => {
    try {
        console.log('Creating product with data:', JSON.stringify(req.body, null, 2));
        // Ensure category is set to 'volcanic' if not provided
        if (!req.body.category) {
            req.body.category = 'volcanic';
        }
        const product = new Product(req.body);
        const savedProduct = await product.save();
        console.log('Product saved successfully:', JSON.stringify({
            _id: savedProduct._id,
            name: savedProduct.name,
            category: savedProduct.category,
            price: savedProduct.price,
            createdAt: savedProduct.createdAt
        }, null, 2));
        res.status(201).json(savedProduct);
    } catch (error) {
        console.error('Error creating product:', error.message);
        res.status(400).json({ error: error.message });
    }
});

// Update product (protected)
app.put('/api/products/:id', requireAuth, async (req, res) => {
    try {
        req.body.updatedAt = Date.now();
        const product = await Product.findByIdAndUpdate(
            req.params.id,
            req.body,
            { new: true, runValidators: true }
        );
        if (!product) {
            return res.status(404).json({ error: 'Product not found' });
        }
        res.json(product);
    } catch (error) {
        res.status(400).json({ error: error.message });
    }
});

// Delete product (protected)
app.delete('/api/products/:id', requireAuth, async (req, res) => {
    try {
        const product = await Product.findByIdAndDelete(req.params.id);
        if (!product) {
            return res.status(404).json({ error: 'Product not found' });
        }
        res.json({ message: 'Product deleted successfully' });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

// Delete all products except volcanic (protected)
app.delete('/api/products/cleanup/non-volcanic', requireAuth, async (req, res) => {
    try {
        const result = await Product.deleteMany({ category: { $ne: 'volcanic' } });
        res.json({ 
            message: 'Non-volcanic products deleted successfully',
            deletedCount: result.deletedCount
        });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

// Delete all products (protected)
app.delete('/api/products/cleanup/all', requireAuth, async (req, res) => {
    try {
        const result = await Product.deleteMany({});
        res.json({ 
            message: 'All products deleted successfully',
            deletedCount: result.deletedCount
        });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

// Health check
app.get('/api/health', (req, res) => {
    res.json({ 
        status: 'OK', 
        database: mongoose.connection.readyState === 1 ? 'Connected' : 'Disconnected' 
    });
});

// Add cache-busting for all HTML files
app.get('*.html', (req, res, next) => {
    res.set({
        'Cache-Control': 'no-store, no-cache, must-revalidate, proxy-revalidate, max-age=0',
        'Pragma': 'no-cache',
        'Expires': '0',
        'Surrogate-Control': 'no-store',
        'Last-Modified': new Date().toUTCString()
    });
    next();
});

// Protected admin route
app.get('/admin', (req, res) => {
    // Disable cache
    res.set({
        'Cache-Control': 'no-store, no-cache, must-revalidate, proxy-revalidate, max-age=0',
        'Pragma': 'no-cache',
        'Expires': '0',
        'Surrogate-Control': 'no-store'
    });
    
    if (req.session && req.session.isAuthenticated) {
        res.sendFile(__dirname + '/public/admin.html');
    } else {
        res.redirect('/login.html');
    }
});

// Function to kill process on port (Windows)
function killProcessOnPort(port) {
    const { exec } = require('child_process');
    return new Promise((resolve) => {
        exec(`netstat -ano | findstr :${port}`, (error, stdout) => {
            if (error || !stdout || stdout.trim() === '') {
                resolve(false);
                return;
            }
            
            const lines = stdout.split('\n').filter(line => line.trim() !== '');
            const pids = new Set();
            
            lines.forEach(line => {
                const parts = line.trim().split(/\s+/);
                const pid = parts[parts.length - 1];
                if (pid && !isNaN(pid)) {
                    pids.add(pid);
                }
            });
            
            if (pids.size === 0) {
                resolve(false);
                return;
            }
            
            if (pids.size === 0) {
                resolve(false);
                return;
            }
            
            let killed = 0;
            const totalPids = pids.size;
            
            pids.forEach(pid => {
                exec(`taskkill /PID ${pid} /F`, (err) => {
                    if (!err) {
                        console.log(`✅ Stopped process ${pid} on port ${port}`);
                    }
                    killed++;
                    if (killed === totalPids) {
                        setTimeout(() => resolve(true), 500);
                    }
                });
            });
        });
    });
}

// Start server with automatic port cleanup
async function startServer() {
    // Try to kill any existing process on the port first
    console.log(`🔍 Checking port ${PORT}...`);
    await killProcessOnPort(PORT);
    
    // Wait a moment for processes to be killed
    await new Promise(resolve => setTimeout(resolve, 1000));
    
    // Start the server
    const server = app.listen(PORT, async () => {
        console.log(`🚀 Server running on http://localhost:${PORT}`);
        console.log(`📦 MongoDB URI: ${MONGODB_URI}`);
        await initializeAdmin();
    });
    
    // Handle port already in use error (fallback)
    server.on('error', async (error) => {
        if (error.code === 'EADDRINUSE') {
            console.log(`\n⚠️  Port ${PORT} is still in use.`);
            console.log(`🔄 Attempting to free port ${PORT} again...`);
            
            await killProcessOnPort(PORT);
            await new Promise(resolve => setTimeout(resolve, 2000));
            
            console.log(`🔄 Retrying to start server...`);
            const newServer = app.listen(PORT, async () => {
                console.log(`🚀 Server running on http://localhost:${PORT}`);
                console.log(`📦 MongoDB URI: ${MONGODB_URI}`);
                await initializeAdmin();
            });
            
            newServer.on('error', (err) => {
                if (err.code === 'EADDRINUSE') {
                    console.error(`\n❌ Port ${PORT} is still in use.`);
                    console.error(`   Please run: npm run kill-port`);
                    console.error(`   Or manually stop the process using port ${PORT}`);
                    process.exit(1);
                } else {
                    throw err;
                }
            });
        } else {
            throw error;
        }
    });
}

// Start the server
startServer();

