// Admin Panel Management with MongoDB API
const API_BASE_URL = window.location.origin;
let editingProductId = null;

// API Functions
async function getProducts() {
    try {
        console.log('Fetching products from:', `${API_BASE_URL}/api/products`);
        const response = await fetch(`${API_BASE_URL}/api/products`, {
            credentials: 'include'
        });
        if (!response.ok) {
            const errorText = await response.text();
            console.error('Failed to fetch products:', response.status, errorText);
            throw new Error(`Failed to fetch products: ${response.status}`);
        }
        const products = await response.json();
        console.log('Products fetched:', products.length, products);
        return products;
    } catch (error) {
        console.error('Error fetching products:', error);
        showMessage('خطأ في تحميل المنتجات', 'error');
        return [];
    }
}

async function createProduct(productData) {
    try {
        console.log('Creating product with data:', productData);
        const response = await fetch(`${API_BASE_URL}/api/products`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            credentials: 'include',
            body: JSON.stringify(productData),
        });
        
        console.log('Response status:', response.status, response.statusText);
        
        if (!response.ok) {
            const errorText = await response.text();
            console.error('Error response:', errorText);
            let errorData;
            try {
                errorData = JSON.parse(errorText);
            } catch {
                errorData = { error: errorText || `HTTP ${response.status}: ${response.statusText}` };
            }
            throw new Error(errorData.error || `HTTP ${response.status}: ${response.statusText}`);
        }
        
        const data = await response.json();
        console.log('Product created successfully:', data);
        return data;
    } catch (error) {
        console.error('Error creating product:', error);
        throw error;
    }
}

async function updateProduct(id, productData) {
    try {
        const response = await fetch(`${API_BASE_URL}/api/products/${id}`, {
            method: 'PUT',
            headers: {
                'Content-Type': 'application/json',
            },
            credentials: 'include',
            body: JSON.stringify(productData),
        });
        
        if (!response.ok) {
            const errorData = await response.json().catch(() => ({}));
            throw new Error(errorData.error || `HTTP ${response.status}: ${response.statusText}`);
        }
        
        const data = await response.json();
        console.log('Product updated:', data);
        return data;
    } catch (error) {
        console.error('Error updating product:', error);
        throw error;
    }
}

async function deleteProduct(id) {
    try {
        const response = await fetch(`${API_BASE_URL}/api/products/${id}`, {
            method: 'DELETE',
            credentials: 'include',
        });
        if (!response.ok) throw new Error('Failed to delete product');
        return await response.json();
    } catch (error) {
        console.error('Error deleting product:', error);
        throw error;
    }
}

// Display products list
async function displayProductsList() {
    const productsList = document.getElementById('productsList');
    const productsCount = document.getElementById('productsCount');
    
    if (!productsList) return;

    // Show loading
    productsList.innerHTML = '<div style="text-align: center; padding: 40px;"><i class="fas fa-spinner fa-spin" style="font-size: 40px; color: var(--primary-teal);"></i></div>';

    try {
        const products = await getProducts();
        console.log('Products loaded for admin:', products.length);

        if (productsCount) {
            productsCount.textContent = `(${products.length})`;
        }

        if (products.length === 0) {
            productsList.innerHTML = `
                <div class="empty-products">
                    <i class="fas fa-box-open"></i>
                    <p>لا توجد منتجات حالياً. أضف منتج جديد للبدء.</p>
                </div>
            `;
            return;
        }

        productsList.innerHTML = '';

        const categoryNames = {
            'volcanic': 'البلك البركاني'
        };

        products.forEach((product) => {
            const productItem = document.createElement('div');
            productItem.className = 'product-item';
            
            productItem.innerHTML = `
                <div class="product-item-image">
                    ${product.image 
                        ? `<img src="${product.image}" alt="${product.name}" onerror="this.parentElement.innerHTML='<div class=\\'placeholder\\'><i class=\\'fas fa-box\\'></i></div>'">`
                        : `<div class="placeholder"><i class="fas fa-box"></i></div>`
                    }
                </div>
                <div class="product-item-info">
                    <h3 class="product-item-name">${product.name}</h3>
                    <span class="product-item-category">${categoryNames[product.category] || product.category}</span>
                    <div class="product-item-price">${product.price} ريال</div>
                </div>
                <div class="product-item-actions">
                    <button class="btn-edit" onclick="editProduct('${product._id}')">
                        <i class="fas fa-edit"></i> تعديل
                    </button>
                    <button class="btn-delete" onclick="deleteProductById('${product._id}')">
                        <i class="fas fa-trash"></i> حذف
                    </button>
                </div>
            `;

            productsList.appendChild(productItem);
        });
    } catch (error) {
        console.error('Error displaying products list:', error);
        productsList.innerHTML = `
            <div class="empty-products">
                <i class="fas fa-exclamation-triangle"></i>
                <p>خطأ في تحميل المنتجات: ${error.message}</p>
                <button onclick="location.reload()" style="margin-top: 10px; padding: 8px 16px; background: var(--primary-teal); color: white; border: none; border-radius: 5px; cursor: pointer;">
                    إعادة المحاولة
                </button>
            </div>
        `;
    }
}

// Add spec input
function addSpecInput() {
    const container = document.getElementById('specsContainer');
    if (!container) return;

    const specItem = document.createElement('div');
    specItem.className = 'spec-item';
    specItem.innerHTML = `
        <input type="text" placeholder="مثال: عزل حراري ممتاز" class="spec-input">
        <button type="button" class="btn-remove-spec" onclick="removeSpecInput(this)">
            <i class="fas fa-times"></i>
        </button>
    `;
    container.appendChild(specItem);
}

// Remove spec input
function removeSpecInput(button) {
    button.parentElement.remove();
}

// Reset form
function resetForm() {
    const form = document.getElementById('productForm');
    if (form) {
        form.reset();
        document.getElementById('specsContainer').innerHTML = '';
        editingProductId = null;
        
        const submitBtn = form.querySelector('.btn-submit');
        if (submitBtn) {
            submitBtn.innerHTML = '<i class="fas fa-save"></i> حفظ المنتج';
        }
    }
}

// Edit product
async function editProduct(id) {
    try {
        const response = await fetch(`${API_BASE_URL}/api/products/${id}`, {
            credentials: 'include'
        });
        if (!response.ok) throw new Error('Failed to fetch product');
        const product = await response.json();
        
        if (!product) return;

        editingProductId = id;

        // Fill form
        document.getElementById('productName').value = product.name || '';
        document.getElementById('productCategory').value = product.category || '';
        document.getElementById('productDescription').value = product.description || '';
        document.getElementById('productPrice').value = product.price || '';
        document.getElementById('productImage').value = product.image || '';

        // Fill specs
        const specsContainer = document.getElementById('specsContainer');
        specsContainer.innerHTML = '';
        if (product.specs && product.specs.length > 0) {
            product.specs.forEach(spec => {
                const specItem = document.createElement('div');
                specItem.className = 'spec-item';
                specItem.innerHTML = `
                    <input type="text" value="${spec}" class="spec-input">
                    <button type="button" class="btn-remove-spec" onclick="removeSpecInput(this)">
                        <i class="fas fa-times"></i>
                    </button>
                `;
                specsContainer.appendChild(specItem);
            });
        }

        // Update submit button
        const submitBtn = document.querySelector('.btn-submit');
        if (submitBtn) {
            submitBtn.innerHTML = '<i class="fas fa-save"></i> تحديث المنتج';
        }

        // Scroll to form
        document.querySelector('.product-form').scrollIntoView({ behavior: 'smooth' });
    } catch (error) {
        console.error('Error editing product:', error);
        showMessage('خطأ في تحميل المنتج', 'error');
    }
}

// Delete product
async function deleteProductById(id) {
    if (!confirm('هل أنت متأكد من حذف هذا المنتج؟')) {
        return;
    }

    try {
        const response = await fetch(`${API_BASE_URL}/api/products/${id}`, {
            method: 'DELETE',
            credentials: 'include',
        });
        
        if (!response.ok) {
            if (response.status === 401) {
                window.location.href = '/login.html';
                return;
            }
            throw new Error('Failed to delete product');
        }
        
        await displayProductsList();
        showMessage('تم حذف المنتج بنجاح', 'success');
    } catch (error) {
        console.error('Delete error:', error);
        showMessage('خطأ في حذف المنتج', 'error');
    }
}

// Show message
function showMessage(message, type = 'info') {
    const messageDiv = document.createElement('div');
    messageDiv.className = `message message-${type}`;
    messageDiv.style.cssText = `
        position: fixed;
        top: 20px;
        left: 20px;
        background: ${type === 'success' ? '#28a745' : type === 'error' ? '#dc3545' : '#17a2b8'};
        color: white;
        padding: 15px 25px;
        border-radius: 5px;
        box-shadow: 0 4px 10px rgba(0,0,0,0.2);
        z-index: 10000;
        animation: slideIn 0.3s ease;
    `;
    messageDiv.textContent = message;
    document.body.appendChild(messageDiv);

    setTimeout(() => {
        messageDiv.style.animation = 'slideOut 0.3s ease';
        setTimeout(() => messageDiv.remove(), 300);
    }, 3000);
}

// Check authentication
async function checkAuth() {
    try {
        const response = await fetch(`${API_BASE_URL}/api/auth/status`, {
            credentials: 'include'
        });
        const data = await response.json();
        
        if (!data.isAuthenticated) {
            window.location.href = '/login.html';
            return false;
        }
        return true;
    } catch (error) {
        console.error('Auth check error:', error);
        window.location.href = '/login.html';
        return false;
    }
}

// Logout function
async function logout() {
    try {
        await fetch(`${API_BASE_URL}/api/auth/logout`, {
            method: 'POST',
            credentials: 'include'
        });
        window.location.href = '/login.html';
    } catch (error) {
        console.error('Logout error:', error);
        window.location.href = '/login.html';
    }
}

// Initialize page
document.addEventListener('DOMContentLoaded', async () => {
    // Check authentication first
    const isAuthenticated = await checkAuth();
    if (!isAuthenticated) return;

    // Display products
    await displayProductsList();

    // Add spec button
    const addSpecBtn = document.getElementById('addSpecBtn');
    if (addSpecBtn) {
        addSpecBtn.addEventListener('click', addSpecInput);
    }

    // Cancel button
    const cancelBtn = document.getElementById('cancelBtn');
    if (cancelBtn) {
        cancelBtn.addEventListener('click', resetForm);
    }

    // Form submit
    const form = document.getElementById('productForm');
    if (form) {
        form.addEventListener('submit', async (e) => {
            e.preventDefault();

            // Get form data
            const name = document.getElementById('productName').value.trim();
            const category = document.getElementById('productCategory').value;
            const description = document.getElementById('productDescription').value.trim();
            const price = parseFloat(document.getElementById('productPrice').value);
            const image = document.getElementById('productImage').value.trim();

            // Get specs
            const specInputs = document.querySelectorAll('.spec-input');
            const specs = Array.from(specInputs)
                .map(input => input.value.trim())
                .filter(spec => spec.length > 0);

            // Validate
            if (!name || !category || !price || price <= 0) {
                showMessage('يرجى ملء جميع الحقول المطلوبة', 'error');
                return;
            }

            // Create product object
            const productData = {
                name,
                category,
                description,
                price: parseFloat(price),
                image,
                specs
            };

            try {
                console.log('Submitting product:', productData);
                let response;
                if (editingProductId) {
                    // Update existing product
                    console.log('Updating product:', editingProductId);
                    response = await updateProduct(editingProductId, productData);
                    if (!response || response.error) {
                        throw new Error(response?.error || 'Failed to update product');
                    }
                    console.log('Product updated:', response);
                    showMessage('تم تحديث المنتج بنجاح', 'success');
                    editingProductId = null; // Reset editing state
                } else {
                    // Add new product
                    console.log('Creating new product...');
                    response = await createProduct(productData);
                    if (!response || response.error) {
                        throw new Error(response?.error || 'Failed to create product');
                    }
                    console.log('Product created:', response);
                    showMessage('تم إضافة المنتج بنجاح', 'success');
                }

                // Wait a bit to ensure product is saved in database
                await new Promise(resolve => setTimeout(resolve, 300));
                
                // Refresh products list immediately
                console.log('Refreshing products list...');
                await displayProductsList();
                console.log('Products list refreshed');
                
                // Scroll to products list after a short delay
                setTimeout(() => {
                    const productsList = document.getElementById('productsList');
                    if (productsList) {
                        productsList.scrollIntoView({ behavior: 'smooth', block: 'start' });
                    }
                }, 500);
                
                resetForm();
            } catch (error) {
                console.error('Save error:', error);
                showMessage(`خطأ في حفظ المنتج: ${error.message}`, 'error');
            }
        });
    }

    // Add CSS animations
    const style = document.createElement('style');
    style.textContent = `
        @keyframes slideIn {
            from {
                transform: translateX(-100%);
                opacity: 0;
            }
            to {
                transform: translateX(0);
                opacity: 1;
            }
        }
        @keyframes slideOut {
            from {
                transform: translateX(0);
                opacity: 1;
            }
            to {
                transform: translateX(-100%);
                opacity: 0;
            }
        }
    `;
    document.head.appendChild(style);
});

// Cleanup functions
async function cleanupNonVolcanic() {
    if (!confirm('هل أنت متأكد من حذف جميع المنتجات ما عدا البلك البركاني؟')) {
        return;
    }

    try {
        const response = await fetch(`${API_BASE_URL}/api/products/cleanup/non-volcanic`, {
            method: 'DELETE',
            credentials: 'include',
        });
        
        if (!response.ok) {
            if (response.status === 401) {
                window.location.href = '/login.html';
                return;
            }
            throw new Error('Failed to cleanup products');
        }
        
        const data = await response.json();
        await displayProductsList();
        showMessage(`تم حذف ${data.deletedCount} منتج بنجاح`, 'success');
    } catch (error) {
        console.error('Cleanup error:', error);
        showMessage('خطأ في حذف المنتجات', 'error');
    }
}

async function cleanupAll() {
    if (!confirm('⚠️ هل أنت متأكد من حذف جميع المنتجات؟\n\nهذه العملية لا يمكن التراجع عنها!')) {
        return;
    }

    if (!confirm('تأكيد نهائي: هل أنت متأكد 100%؟')) {
        return;
    }

    try {
        const response = await fetch(`${API_BASE_URL}/api/products/cleanup/all`, {
            method: 'DELETE',
            credentials: 'include',
        });
        
        if (!response.ok) {
            if (response.status === 401) {
                window.location.href = '/login.html';
                return;
            }
            throw new Error('Failed to cleanup products');
        }
        
        const data = await response.json();
        await displayProductsList();
        showMessage(`تم حذف ${data.deletedCount} منتج بنجاح`, 'success');
    } catch (error) {
        console.error('Cleanup error:', error);
        showMessage('خطأ في حذف المنتجات', 'error');
    }
}

// Export functions
window.logout = logout;
window.cleanupNonVolcanic = cleanupNonVolcanic;
window.cleanupAll = cleanupAll;

