// Product Details Management
const API_BASE_URL = window.location.origin;

// Get product ID from URL
function getProductIdFromURL() {
    const urlParams = new URLSearchParams(window.location.search);
    return urlParams.get('id');
}

// Fetch product details
async function getProductDetails(productId) {
    try {
        const response = await fetch(`${API_BASE_URL}/api/products/${productId}`);
        if (!response.ok) {
            throw new Error('Failed to fetch product');
        }
        return await response.json();
    } catch (error) {
        console.error('Error fetching product:', error);
        throw error;
    }
}

// Display product details
async function displayProductDetails() {
    const content = document.getElementById('productDetailsContent');
    const productId = getProductIdFromURL();
    
    if (!productId) {
        content.innerHTML = `
            <div class="error-state">
                <i class="fas fa-exclamation-triangle"></i>
                <h2>منتج غير موجود</h2>
                <p>لم يتم العثور على المنتج المطلوب</p>
                <a href="products.html" class="btn-retry">العودة إلى المنتجات</a>
            </div>
        `;
        return;
    }
    
    try {
        const product = await getProductDetails(productId);
        
        const categoryNames = {
            'volcanic': 'البلك البركاني'
        };
        
        const specsHTML = product.specs && product.specs.length > 0
            ? `<ul class="product-details-specs">
                ${product.specs.map(spec => `<li><i class="fas fa-check"></i> ${spec}</li>`).join('')}
               </ul>`
            : '';
        
        content.innerHTML = `
            <a href="products.html" class="back-button">
                <i class="fas fa-arrow-right"></i> العودة إلى المنتجات
            </a>
            <div class="product-details-container">
                <div class="product-details-top">
                    <div class="product-details-image">
                        ${product.image 
                            ? `<img src="${product.image}" alt="${product.name}" onerror="this.parentElement.innerHTML='<div class=\\'placeholder\\'><i class=\\'fas fa-box\\'></i></div>'">`
                            : `<div class="placeholder"><i class="fas fa-box"></i></div>`
                        }
                    </div>
                    <div class="product-details-left">
                        <div class="price-section">
                            <div class="price-label">السعر للحبة الواحدة</div>
                            <div class="product-details-price">${product.price} ر.س</div>
                            <div class="price-vat">شامل ضريبة القيمة المضافة</div>
                        </div>
                        <button class="btn-contact-details" onclick="contactAboutProduct('${product.name}')">
                            <i class="fas fa-phone"></i> اتصل بنا
                        </button>
                    </div>
                </div>
                <div class="product-details-description-section">
                    <span class="product-details-category">${categoryNames[product.category] || product.category}</span>
                    <h1 class="product-details-name">${product.name}</h1>
                    ${product.description ? `<p class="product-details-description">${product.description}</p>` : ''}
                    ${specsHTML}
                </div>
            </div>
        `;
    } catch (error) {
        console.error('Error displaying product:', error);
        content.innerHTML = `
            <div class="error-state">
                <i class="fas fa-exclamation-triangle"></i>
                <h2>حدث خطأ</h2>
                <p>تعذر تحميل تفاصيل المنتج</p>
                <button class="btn-retry" onclick="location.reload()">إعادة المحاولة</button>
            </div>
        `;
    }
}

// Contact about product
function contactAboutProduct(productName) {
    const message = `أريد الاستفسار عن: ${productName}`;
    const phone = '+966561033147';
    const whatsappLink = `https://wa.me/${phone.replace(/\D/g, '')}?text=${encodeURIComponent(message)}`;
    window.open(whatsappLink, '_blank');
}

// Initialize page
document.addEventListener('DOMContentLoaded', () => {
    displayProductDetails();
});

